/**
 * SEO and URL utilities
 * Ensures sensitive parameters are never exposed to search engines
 */

// Google Ads parameter names that must NEVER be indexed
const SENSITIVE_PARAMS = [
  "gclid",
  "gad_source",
  "gad_campaignid",
  "gbraid",
  "wbraid",
  "gclsrc",
] as const;

/**
 * Strips all sensitive parameters from a URL or URL string
 * Used for canonical URLs, Open Graph tags, and any indexable content
 */
export function stripSensitiveParams(url: string | URL): string {
  const urlObj = typeof url === "string" ? new URL(url) : url;
  
  // Remove all sensitive parameters
  SENSITIVE_PARAMS.forEach((param) => {
    urlObj.searchParams.delete(param);
  });
  
  return urlObj.toString();
}

/**
 * Checks if a URL contains any sensitive parameters
 */
export function hasSensitiveParams(url: string | URL): boolean {
  const urlObj = typeof url === "string" ? new URL(url) : url;
  
  return SENSITIVE_PARAMS.some((param) => urlObj.searchParams.has(param));
}

/**
 * Creates a clean canonical URL without sensitive parameters
 */
export function createCanonicalUrl(path: string, baseUrl: string): string {
  const url = new URL(path, baseUrl);
  return stripSensitiveParams(url);
}

