import { UAParser } from "ua-parser-js";

const BOT_USER_AGENT_KEYWORDS = [
  "googlebot",
  "adsbot-google",
  "bingbot",
  "slurp",
  "duckduckbot",
  "baiduspider",
  "yandexbot",
  "sogou",
  "exabot",
  "facebot",
  "ia_archiver",
];

// Google Ads parameter names to detect
const GOOGLE_ADS_PARAMS = [
  "gclid",
  "gad_source",
  "gad_campaignid",
  "gbraid",
  "wbraid",
  "gclsrc",
];

export function isBot(userAgent: string | null, visitorTypeHeader?: string | null) {
  if (visitorTypeHeader === "bot") return true;
  if (!userAgent) return true;
  const normalized = userAgent.toLowerCase();
  return BOT_USER_AGENT_KEYWORDS.some((keyword) => normalized.includes(keyword));
}

export function isMobileDevice(userAgent: string | null) {
  if (!userAgent) return false;
  const parser = new UAParser(userAgent);
  const deviceType = parser.getDevice().type;
  return deviceType === "mobile" || deviceType === "tablet";
}

/**
 * Checks if the user comes from Google Ads by detecting Google Ads parameters in the URL
 */
export function hasGoogleAdsParams(searchParams: Record<string, string | string[] | undefined>): boolean {
  return GOOGLE_ADS_PARAMS.some((param) => {
    const value = searchParams[param];
    return value !== undefined && value !== null && value !== "";
  });
}

/**
 * Checks if the referer indicates the user came from Google Search
 * The referer should contain "google.com/search" or have google.com hostname with /search path
 */
export function isFromGoogleSearch(referer: string | null): boolean {
  if (!referer) return false;
  
  const refererLower = referer.toLowerCase();
  
  // Check if referer contains "google.com/search" (most common case)
  if (refererLower.includes("google.com/search")) {
    return true;
  }
  
  // Also check for valid URL with google.com hostname and /search path
  try {
    const refererUrl = new URL(referer);
    const hostname = refererUrl.hostname.toLowerCase();
    const pathname = refererUrl.pathname.toLowerCase();
    
    // Check if hostname is google.com (or www.google.com) and path contains /search
    const isGoogleHost = hostname === "google.com" || hostname === "www.google.com";
    const hasSearchPath = pathname.includes("/search");
    
    return isGoogleHost && hasSearchPath;
  } catch {
    // If referer is not a valid URL, fallback to simple string check
    return refererLower.includes("google.com") && refererLower.includes("search");
  }
}

type ShouldShowAllPartnersArgs = {
  userAgent: string | null;
  visitorTypeHeader?: string | null;
  searchParams: Record<string, string | string[] | undefined>;
  referer: string | null;
};

/**
 * RULE 1: Display BOTH Licensed and Unlicensed Brands ONLY if ALL conditions are TRUE:
 * 1. User comes from Google Ads (gclid, gad_source, gad_campaignid, etc.)
 * 2. User is on mobile device
 * 3. User arrives from Google Search (referer contains "google.com/search")
 * 4. User is a real human (not a bot)
 * 
 * RULE 2: Otherwise, show ONLY Licensed Brands
 */
export function shouldShowAllPartners({
  userAgent,
  visitorTypeHeader,
  searchParams,
  referer,
}: ShouldShowAllPartnersArgs): boolean {
  // Condition 1: Check if user comes from Google Ads
  const hasGoogleAds = hasGoogleAdsParams(searchParams);
  
  // Condition 2: Check if user is on mobile device
  const isMobile = isMobileDevice(userAgent);
  
  // Condition 3: Check if user arrives from Google Search
  const fromGoogleSearch = isFromGoogleSearch(referer);
  
  // Condition 4: Check if user is a real human (not a bot)
  const isHuman = !isBot(userAgent, visitorTypeHeader);
  
  // Show ALL partners (licensed + unlicensed) ONLY if ALL conditions are met
  return hasGoogleAds && isMobile && fromGoogleSearch && isHuman;
}
