"use client";

import { useState, useEffect } from "react";
import type { SiteInfo } from "@/types/data";

type SiteInfoTabProps = {
  siteInfo: SiteInfo | null;
  onUpdate: () => void;
};

export function SiteInfoTab({ siteInfo, onUpdate }: SiteInfoTabProps) {
  const [formData, setFormData] = useState<SiteInfo>({
    siteName: "",
    siteLogo: "",
    email: "",
    address: "",
    phone: "",
    accountCode: "",
    tagline: "",
    favicon: "",
    browserTitle: "",
  });

  useEffect(() => {
    if (siteInfo) {
      setFormData(siteInfo);
    }
  }, [siteInfo]);

  const handleSave = async () => {
    try {
      await fetch("/api/admin/siteinfo", {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ 
          siteInfo: formData,
          oldSiteLogo: siteInfo?.siteLogo,
          oldFavicon: siteInfo?.favicon
        }),
      });
      alert("Site info updated successfully!");
      onUpdate();
    } catch (error) {
      alert("Failed to update site info");
    }
  };

  if (!siteInfo) {
    return <div>Loading...</div>;
  }

  return (
    <div className="space-y-6">
      <div className="flex justify-between">
        <h2 className="text-2xl font-bold text-slate-900">Site Information</h2>
        <button
          onClick={handleSave}
          className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-semibold text-white transition-colors hover:bg-blue-700"
        >
          Save Changes
        </button>
      </div>

      <div className="rounded-xl border border-slate-200 bg-white p-6 shadow-sm">
        <div className="grid gap-6 sm:grid-cols-2">
          <div>
            <label className="block text-sm font-medium text-slate-700">
              Site Name
            </label>
            <input
              type="text"
              value={formData.siteName}
              onChange={(e) =>
                setFormData({ ...formData, siteName: e.target.value })
              }
              className="mt-1 block w-full rounded-lg border border-slate-300 px-3 py-2 text-slate-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-slate-700">
              Site Logo
            </label>
            <div className="mt-1 space-y-2">
              <input
                type="file"
                accept="image/*"
                onChange={async (e) => {
                  const file = e.target.files?.[0];
                  if (file) {
                    const uploadFormData = new FormData();
                    uploadFormData.append('file', file);
                    uploadFormData.append('type', 'site');
                    // Pass old logo path if exists
                    if (formData.siteLogo) {
                      uploadFormData.append('oldPath', formData.siteLogo);
                    }
                    
                    try {
                      const res = await fetch('/api/admin/upload', {
                        method: 'POST',
                        body: uploadFormData,
                      });
                      const data = await res.json();
                      if (data.success) {
                        setFormData({ ...formData, siteLogo: data.path });
                      } else {
                        alert('Failed to upload logo');
                      }
                    } catch (error) {
                      alert('Failed to upload logo');
                    }
                  }
                }}
                className="block w-full text-sm text-slate-600 file:mr-4 file:rounded-lg file:border-0 file:bg-blue-50 file:px-4 file:py-2 file:text-sm file:font-semibold file:text-blue-700 hover:file:bg-blue-100"
              />
              <input
                type="text"
                value={formData.siteLogo}
                onChange={(e) =>
                  setFormData({ ...formData, siteLogo: e.target.value })
                }
                className="block w-full rounded-lg border border-slate-300 px-3 py-2 text-slate-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                placeholder="/logo.png"
              />
            </div>
          </div>
          <div>
            <label className="block text-sm font-medium text-slate-700">
              Email
            </label>
            <input
              type="email"
              value={formData.email}
              onChange={(e) =>
                setFormData({ ...formData, email: e.target.value })
              }
              className="mt-1 block w-full rounded-lg border border-slate-300 px-3 py-2 text-slate-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-slate-700">
              Phone
            </label>
            <input
              type="text"
              value={formData.phone}
              onChange={(e) =>
                setFormData({ ...formData, phone: e.target.value })
              }
              className="mt-1 block w-full rounded-lg border border-slate-300 px-3 py-2 text-slate-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-slate-700">
              Address
            </label>
            <input
              type="text"
              value={formData.address}
              onChange={(e) =>
                setFormData({ ...formData, address: e.target.value })
              }
              className="mt-1 block w-full rounded-lg border border-slate-300 px-3 py-2 text-slate-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-slate-700">
              Account Code
            </label>
            <input
              type="text"
              value={formData.accountCode}
              onChange={(e) =>
                setFormData({ ...formData, accountCode: e.target.value })
              }
              className="mt-1 block w-full rounded-lg border border-slate-300 px-3 py-2 text-slate-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
            />
          </div>
          <div className="sm:col-span-2">
            <label className="block text-sm font-medium text-slate-700">
              Tagline (Header & Homepage)
            </label>
            <input
              type="text"
              value={formData.tagline}
              onChange={(e) =>
                setFormData({ ...formData, tagline: e.target.value })
              }
              className="mt-1 block w-full rounded-lg border border-slate-300 px-3 py-2 text-slate-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
              placeholder="Top-Rated UK Betting Platforms"
            />
            <p className="mt-1 text-xs text-slate-500">
              This appears in the header subtitle and homepage heading
            </p>
          </div>
          <div>
            <label className="block text-sm font-medium text-slate-700">
              Browser Title
            </label>
            <input
              type="text"
              value={formData.browserTitle}
              onChange={(e) =>
                setFormData({ ...formData, browserTitle: e.target.value })
              }
              className="mt-1 block w-full rounded-lg border border-slate-300 px-3 py-2 text-slate-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
              placeholder="Top-Rated UK Betting Platforms"
            />
            <p className="mt-1 text-xs text-slate-500">
              This appears in the browser tab
            </p>
          </div>
          <div>
            <label className="block text-sm font-medium text-slate-700">
              Favicon
            </label>
            <div className="mt-1 space-y-2">
              <input
                type="file"
                accept="image/*,.ico"
                onChange={async (e) => {
                  const file = e.target.files?.[0];
                  if (file) {
                    const uploadFormData = new FormData();
                    uploadFormData.append('file', file);
                    uploadFormData.append('type', 'favicon');
                    // Pass old favicon path if exists
                    if (formData.favicon) {
                      uploadFormData.append('oldPath', formData.favicon);
                    }
                    
                    try {
                      const res = await fetch('/api/admin/upload', {
                        method: 'POST',
                        body: uploadFormData,
                      });
                      const data = await res.json();
                      if (data.success) {
                        setFormData({ ...formData, favicon: data.path });
                      } else {
                        alert('Failed to upload favicon');
                      }
                    } catch (error) {
                      alert('Failed to upload favicon');
                    }
                  }
                }}
                className="block w-full text-sm text-slate-600 file:mr-4 file:rounded-lg file:border-0 file:bg-blue-50 file:px-4 file:py-2 file:text-sm file:font-semibold file:text-blue-700 hover:file:bg-blue-100"
              />
              <input
                type="text"
                value={formData.favicon}
                onChange={(e) =>
                  setFormData({ ...formData, favicon: e.target.value })
                }
                className="block w-full rounded-lg border border-slate-300 px-3 py-2 text-slate-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                placeholder="/favicon.ico"
              />
            </div>
            <p className="mt-1 text-xs text-slate-500">
              Upload a .ico, .png, or .svg file for the browser favicon
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}

