const { createServer } = require('http');
const { parse } = require('url');
const next = require('next');

// Error handling for uncaught exceptions
process.on('uncaughtException', (err) => {
  console.error('Uncaught Exception:', err);
  process.exit(1);
});

process.on('unhandledRejection', (reason, promise) => {
  console.error('Unhandled Rejection at:', promise, 'reason:', reason);
  process.exit(1);
});

// Check if .next directory exists
const fs = require('fs');
const path = require('path');
const nextBuildPath = path.join(process.cwd(), '.next');

if (!fs.existsSync(nextBuildPath)) {
  console.error('ERROR: .next directory not found!');
  console.error('Please run "npm run build" first to create the production build.');
  process.exit(1);
}

const dev = process.env.NODE_ENV !== 'production';
const hostname = process.env.HOSTNAME || '0.0.0.0'; // Changed to 0.0.0.0 for hosting compatibility
const port = parseInt(process.env.PORT || '3000', 10);

console.log(`Starting Next.js server...`);
console.log(`Environment: ${dev ? 'development' : 'production'}`);
console.log(`Hostname: ${hostname}`);
console.log(`Port: ${port}`);

// when using middleware `hostname` and `port` must be provided below
const app = next({ dev, hostname, port });
const handle = app.getRequestHandler();

app.prepare()
  .then(() => {
    console.log('Next.js app prepared successfully');
    
    const server = createServer(async (req, res) => {
      try {
        // Be sure to pass true as the second argument to url.parse.
        // This tells it to parse the query portion of the URL.
        const parsedUrl = parse(req.url, true);
        const { pathname, query } = parsedUrl;

        if (pathname === '/a') {
          await app.render(req, res, '/a', query);
        } else if (pathname === '/b') {
          await app.render(req, res, '/b', query);
        } else {
          await handle(req, res, parsedUrl);
        }
      } catch (err) {
        console.error('Error occurred handling', req.url, err);
        res.statusCode = 500;
        res.end('internal server error');
      }
    });

    server.once('error', (err) => {
      console.error('Server error:', err);
      if (err.code === 'EADDRINUSE') {
        console.error(`Port ${port} is already in use`);
      }
      process.exit(1);
    });

    server.listen(port, hostname, () => {
      console.log(`> Ready on http://${hostname}:${port}`);
      console.log(`> Server started successfully`);
    });
  })
  .catch((err) => {
    console.error('Failed to prepare Next.js app:', err);
    console.error('Error details:', err.stack);
    process.exit(1);
  });
